package Musings::Article;

use Carp qw(croak);
use Musings::DB;
use strict;

=head1 NAME

Musings::Article

=head1 SYNOPSIS

  use Musings::Article;
  my $article = Musings::Article->new;
  $article->article_title('History of librarianship');
  $article->article_note('This article is about...');
  $article->article_content('A long time ago, in a galaxy...');
  $article->article_changes('I changed this, then that...');
  $article->article_date('2004-11-20');
  $article->article_path('/musings/librarianship/');
  $article->article_source('Originally published in...');
  $article->article_filename('librarianship.xml');
  $article->template_id('5');
  $article->stylesheet_id('13');
  $article->author_ids(['4', '7', '18', '9']);
  $article->subject_ids(['2', '5', '11']);
  $article->commit;
  $article = Musings::Article->new(id => $id);
  my $article_title    = $article->article_title;
  my $article_note     = $article->article_note;
  my $article_content  = $article->article_content;
  my $article_changes  = $article->article_changes;
  my $article_date     = $article->article_date;
  my $article_path     = $article->article_path;
  my $article_source   = $article->article_source;
  my $article_filename = $article->article_filename;
  my $template_id      = $article->template_id;
  my $stylesheet_id    = $article->stylesheet_id;
  my @author_ids       = $article->author_ids;
  my @subject_ids      = $article->subject_ids;
  $article->delete;
  my @articles = $article->get_articles;


=head1 DESCRIPTION

This package does the necessary I/O against the underlying database's articles table.


=head1 METHODS


=head2 new

Creates a new article object. If passed an id parameter plus a valid article ID, then a specific article object will be created.


=head2 article_title

Get and set the article's title.


=head2 article_note

Get and set the article's note. This is intended to be analogous to the Dublin Core description -- an abstract.


=head2 article_content

Get and set the content of the article. It is intended to be everything inside the body element of an HTML file or TEI file. This content is expected to be marked-up accordingly. This module is not intended to be an editor.


=head2 article_changes

Get and set the text describing how the document has been changed over time. This value is intended to be the valid content for the revisionDesc element of a TEI file. You are expected to include the necessary mark-up.


=head2 article_date

Get and set the date this article was originally created. The date should be in the form of YYYY-MM-DD.


=head2 article_path

Get and set the path where the document is expected to be saved to a file system. 


=head2 article_filename

Get and set the name of the file when (and if) you save it to the file system.


=head2 template_id

Get and set the template ID associated with this article. One template per article.


=head2 stylesheet_id

Get and set the stylesheet ID associated with this article. One stylesheet per article.


=head2 author_ids

Get and set the author ID's associated with this article. To set the ID's pass a referenced array. The result of get operation will be a simple array of integers.


=head2 subject_ids

Get and set the subjectg ID's associated with this article. To set the ID's pass a referenced array. The result of get operation will be a simple array of integers.


=head2 commit

Save the current article object's attributes to the database. Returns true upon success.


=head2 article_id

Gets the current ID of the current article object. You can not set the article_id value.


=head2 delete

Deletes (drops) the current article from the database. Be careful. Your application is responsible for passing the ACID test.


=head2 get_articles

Returns an array of all the Musings::Article objects in the database. The returned array will be sorted by article_title.


=head1 AUTHORS

Eric Lease Morgan <eric_morgan@infomotions.com>

=cut



sub new {

	# declare local variables
	my ($class, %opts) = @_;
	my $self           = {};
	
	# check for an id
	if ($opts{id}) {
	
		# get a handle
		my $dbh = Musings::DB->dbh();
		
		# find this record
		my $rv = $dbh->selectrow_hashref('SELECT * FROM articles WHERE article_id = ?', undef, $opts{id});
		
		# check for success
		if (ref($rv) eq "HASH") {
		
			# fill the object
			$self = $rv;
			
			# build a list of author ids
			my @author_ids = ();
			my $ids = $dbh->selectall_arrayref('SELECT author_id FROM items4authors WHERE article_id = ?', undef, $opts{id});
			foreach (@{$ids}) { push (@author_ids, $_->[0]) }
			$self->{author_ids} = ([@author_ids]);
			
			# build a list of subject ids
			my @subject_ids = ();
			$ids = $dbh->selectall_arrayref('SELECT subject_id FROM items4subjects WHERE article_id = ?', undef, $opts{id});
			foreach (@{$ids}) { push (@subject_ids, $_->[0]) }
			$self->{subject_ids} = ([@subject_ids]);
			
		}
		
		else { return }
	
	} 
	
	# return the object
	return bless $self, $class;
	
}


sub article_id {

	my $self = shift;
	return $self->{'article_id'};

}


sub article_title {

	my ($self, $article_title) = @_;
	
	if ($article_title) { $self->{article_title} = $article_title }
	else { return $self->{article_title} }
	
}


sub article_changes {

	my ($self, $article_changes) = @_;
	
	if ($article_changes) { $self->{article_changes} = $article_changes }
	else { return $self->{article_changes} }
	
}


sub article_content {

	my ($self, $article_content) = @_;
	
	if ($article_content) { $self->{article_content} = $article_content }
	else { return $self->{article_content} }
	
}


sub article_contributor {

	my ($self, $article_contributor) = @_;
	
	if ($article_contributor) { $self->{article_contributor} = $article_contributor }
	else { return $self->{article_contributor} }
	
}


sub article_date {

	my ($self, $article_date) = @_;
	
	if ($article_date) { $self->{article_date} = $article_date }
	else { return $self->{article_date} }
	
}


sub article_note {

	my ($self, $article_note) = @_;
	
	if ($article_note) { $self->{article_note} = $article_note }
	else { return $self->{article_note} }
	
}


sub article_path {

	my ($self, $article_path) = @_;
	
	if ($article_path) { $self->{article_path} = $article_path }
	else { return $self->{article_path} }
	
}


sub article_source {

	my ($self, $article_source) = @_;
	
	if ($article_source) { $self->{article_source} = $article_source }
	else { return $self->{article_source} }
	
}

sub article_filename {

	my ($self, $article_filename) = @_;
	
	if ($article_filename) { $self->{article_filename} = $article_filename }
	else { return $self->{article_filename} }
	
}


sub template_id {

	my ($self, $template_id) = @_;
	
	if ($template_id) { $self->{template_id} = $template_id }
	else { return $self->{template_id} }
	
}

sub stylesheet_id {

	my ($self, $stylesheet_id) = @_;
	
	if ($stylesheet_id) { $self->{stylesheet_id} = $stylesheet_id }
	else { return $self->{stylesheet_id} }
	
}



sub commit {

	# get myself, :-)
	my $self = shift;
	
	# get a database handle
	my $dbh = Musings::DB->dbh();	
	
	# see if the object has an id
	if ($self->article_id()) {
	
		# update the record with this id
		my $return = $dbh->do('UPDATE articles SET article_title = ?, article_changes = ?, article_content = ?, article_contributor = ?, article_date = ?, article_note = ?, article_path = ?, article_source = ?, stylesheet_id = ?, template_id = ?, article_filename = ? WHERE article_id = ?', undef, $self->article_title(), $self->article_changes(), $self->article_content(), $self->article_contributor(), $self->article_date(), $self->article_note(), $self->article_path(), $self->article_source(), $self->stylesheet_id(), $self->template_id(), $self->article_filename(), $self->article_id());
		if ($return > 1 || ! $return) { croak "Article update in commit() failed. $return records were updated." }
	
		# delete the existing relationships
		$return = $dbh->do('DELETE FROM items4authors WHERE article_id = ?', undef, $self->article_id());
		if (! $return) { croak 'Could not delete author/article relationships.'; }

		# delete the existing relationships
		$return = $dbh->do('DELETE FROM items4subjects WHERE article_id = ?', undef, $self->article_id());
		if (! $return) { croak 'Could not delete subject/article relationships.'; }

		# create author relationships
		foreach (@{$self->{author_ids}}) {
		
			my $return = $dbh->do('INSERT INTO items4authors (article_id, author_id) VALUES (?, ?)', undef, $self->article_id(), $_);
			if ($return > 1) { croak 'Article/author commit() failed.'; }
		
		}

		# create subject relationships
		foreach (@{$self->{subject_ids}}) {
		
			my $return = $dbh->do('INSERT INTO items4subjects (article_id, subject_id) VALUES (?, ?)', undef, $self->article_id(), $_);
			if ($return > 1) { croak 'Article/subject commit() failed.'; }
		
		}

	}
	
	else {
	
		# get a new sequence
		my $id = Musings::DB->nextID();
				
		# create a new record
		my $return = $dbh->do('INSERT INTO articles (article_id, article_title, article_changes, article_content, article_contributor, article_date, article_note, article_path, article_source, stylesheet_id, template_id, article_filename) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)', undef, $id, $self->article_title(), $self->article_changes(), $self->article_content(), $self->article_contributor(), $self->article_date(), $self->article_note(), $self->article_path(), $self->article_source(), $self->stylesheet_id(), $self->template_id(), $self->article_filename());
		if ($return > 1) { croak 'Article commit() failed.'; }
		$self->{article_id} = $id;
		
		# create author relationships
		foreach (@{$self->{author_ids}}) {
		
			my $return = $dbh->do('INSERT INTO items4authors (article_id, author_id) VALUES (?, ?)', undef, $id, $_);
			if ($return > 1) { croak 'Article/author commit() failed.'; }
		
		}
		
		# create subject relationships
		foreach (@{$self->{subject_ids}}) {
		
			my $return = $dbh->do('INSERT INTO items4subjects (article_id, subject_id) VALUES (?, ?)', undef, $id, $_);
			if ($return > 1) { croak 'Article/subject commit() failed.'; }
		
		}
		
	}
	
	# done
	return 1;
	
}


sub delete {

	my $self = shift;

	if ($self->{article_id}) {

		my $dbh = Musings::DB->dbh();

		# delete the existing relationships
		my $return = $dbh->do('DELETE FROM items4authors WHERE article_id = ?', undef, $self->article_id());
		if (! $return) { croak 'Could not delete author/article relationships.'; }

		# delete the existing relationships
		$return = $dbh->do('DELETE FROM items4subjects WHERE article_id = ?', undef, $self->article_id());
		if (! $return) { croak 'Could not delete subject/article relationships.'; }

		# now, delete the primary facet record
		my $rv = $dbh->do('DELETE FROM articles WHERE article_id = ?', undef, $self->{article_id});
		if ($rv != 1) {croak ("Error deleting article record. Deleted $rv records.");}
		return 1;

	}

	return 0;

}


sub get_articles {

	my $self = shift;
	my %opts = @_;
	my @rv   = ();

	my ($sort, $field, $value, $sort_clause, $limit_clause, $query);
	if (defined($opts{sort})) {
	
		if ($opts{sort} eq 'name') { $sort_clause = 'ORDER BY article_title' }
		if ($opts{sort} eq 'date') { $sort_clause = 'ORDER BY article_date DESC' }
		
	}
	
	if (defined($opts{field}) && defined($opts{value})) {
	
		$field = $opts{'field'};
		$value = $opts{'value'};
		if ($field eq 'name') { $limit_clause = "WHERE article_title LIKE \'%$value%\'" }
		elsif ($field eq 'description') { $limit_clause = "WHERE article_note LIKE \'%$value%\'" }
		
	}
	
	$query = 'SELECT article_id FROM articles';
	if ($limit_clause) { $query .= " $limit_clause" }
	if ($sort_clause) { $query .= " $sort_clause" }

	# create and execute a query
	my $dbh = Musings::DB->dbh();

	my $article_ids = $dbh->selectcol_arrayref("$query");
			
	foreach my $article_id (@$article_ids) { push (@rv, Musings::Article->new(id => $article_id)) }	
	
	return @rv;
	
}


sub author_ids {

	# get the input
	my $self       = shift;
	my @author_ids = @_;
								
	# check for input
	if (@author_ids) { $self->{author_ids} = ([@author_ids]) }
		
	else {
	
		# rebuild author_ids
		@author_ids = ();
		foreach (@{$self->{author_ids}}) { push(@author_ids, $_) }
		
	}
		
	# done
	return @author_ids;
	
}


sub subject_ids {

	# get the input
	my $self       = shift;
	my @subject_ids = @_;
								
	# check for input
	if (@subject_ids) { $self->{subject_ids} = ([@subject_ids]) }
		
	else {
	
		# rebuild subject_ids
		@subject_ids = ();
		foreach (@{$self->{subject_ids}}) { push(@subject_ids, $_) }
		
	}
		
	# done
	return @subject_ids;
	
}


# return true
1;
